import { renderMatrix } from './modules/render.js';
import { updateUI } from './modules/ui.js';
import { initScrollFx } from './modules/scrollFx.js';

// Глобальные переменные
let globalNodes = [], globalMeta = {}, currentMode = 'single';

// === 1. НОВАЯ ФИЗИКА ШТОРКИ (FREE DRAG) ===
class SheetPhysics {
    constructor() {
        this.sheet = document.getElementById("bottomSheet");
        this.header = document.getElementById("sheetHeader");
        // Стартовая позиция: 85% отступа сверху (торчит 15% снизу)
        this.currentPercent = 85; 
        
        if(this.sheet) {
            this.sheet.style.transform = `translateY(${this.currentPercent}%)`;
        }
        this.init();
    }

    init() {
        if(!this.sheet || !this.header) return;

        let startY = 0;
        let startPercent = 0;
        let isDragging = false;

        const start = (y) => {
            isDragging = true;
            startY = y;
            startPercent = this.currentPercent;
            this.sheet.style.transition = "none"; // Убираем анимацию при драге
        };

        const move = (y) => {
            if (!isDragging) return;
            const deltaY = y - startY;
            const screenH = window.innerHeight;
            const deltaPercent = (deltaY / screenH) * 100;
            
            let newPercent = startPercent + deltaPercent;

            // Ограничители (0 = верх, 92 = низ)
            if (newPercent < 0) newPercent = 0;
            if (newPercent > 92) newPercent = 92;

            this.currentPercent = newPercent;
            this.sheet.style.transform = `translateY(${newPercent}%)`;
        };

        const end = () => {
            if (!isDragging) return;
            isDragging = false;
            // Плавная остановка, но без привязки к точкам
            this.sheet.style.transition = "transform 0.2s ease-out";
        };

        this.header.onmousedown = e => start(e.clientY);
        document.onmousemove = e => move(e.clientY);
        document.onmouseup = end;
        this.header.ontouchstart = e => start(e.touches[0].clientY);
        document.ontouchmove = e => { if(isDragging) move(e.touches[0].clientY); };
        document.ontouchend = end;
    }
    
    // Метод для программного открытия (например, при расчете)
    openHalf() {
        this.currentPercent = 50;
        if(this.sheet) {
            this.sheet.style.transition = "transform 0.3s ease-out";
            this.sheet.style.transform = `translateY(50%)`;
        }
    }
}

// === 2. СЛЕЖЕНИЕ ЗА СЛАЙДАМИ ===
function initSliderWatcher() {
    const box = document.getElementById('slidesContainer');
    if(!box) return null;
    const obs = new IntersectionObserver(es => {
        es.forEach(e => {
            if(e.isIntersecting) {
                const k = e.target.dataset.key;
                document.querySelectorAll('.tab-item').forEach(t => {
                    if(t.dataset.key===k) { 
                        t.classList.add('active'); 
                        t.scrollIntoView({behavior:'smooth',inline:'center'}); 
                    } else t.classList.remove('active');
                });
            }
        });
    }, {root:box, threshold:0.5});
    return obs;
}

// === 3. TERMINAL DATE LOGIC ===

// Данные для генерации
const DATES_CFG = {
    day: { start: 1, end: 31, pad: true },
    month: { start: 1, end: 12, pad: true },
    year: { start: 1960, end: 2030, pad: false }
};

// Функция создания выпадающего списка
function initTermInput(id, type) {
    const el = document.getElementById(id);
    if (!el) return;

    // 1. Берем текущее значение и чистим элемент полностью
    const def = el.getAttribute('data-val') || el.innerText.trim();
    el.innerHTML = ''; // Важно: очищаем от старого мусора
    el.dataset.val = def;

    // 2. Создаем SPAN для отображения цифры (чтобы список не стирался)
    const valSpan = document.createElement('span');
    valSpan.className = 'term-val-text';
    valSpan.innerText = def;
    el.appendChild(valSpan);

    // 3. Генерируем список
    const list = document.createElement('div');
    list.className = 'term-dropdown';
    
    const cfg = DATES_CFG[type];
    for(let i = cfg.start; i <= cfg.end; i++) {
        const val = cfg.pad && i < 10 ? '0' + i : '' + i;
        const opt = document.createElement('div');
        opt.className = 'term-option';
        opt.innerText = val;
        
        // Клик по опции
        opt.onclick = (e) => {
            e.stopPropagation();
            // Обновляем ТОЛЬКО текст в span и атрибут, список жив
            valSpan.innerText = val; 
            el.dataset.val = val; 
            closeAllTerm();
        };
        list.appendChild(opt);
    }
    
    el.appendChild(list);

    // 4. Клик по самому полю
    el.onclick = (e) => {
        e.stopPropagation();
        const isOpen = list.classList.contains('show');
        closeAllTerm();
        if(!isOpen) {
            list.classList.add('show');
            el.classList.add('active');
            
            // Авто-скролл к выбранному
            const currentVal = el.dataset.val;
            const opts = Array.from(list.children);
            const target = opts.find(o => o.innerText === currentVal);
            if(target) target.scrollIntoView({block: "center"});
        }
    };
}

// Закрытие всех списков
function closeAllTerm() {
    document.querySelectorAll('.term-dropdown').forEach(d => d.classList.remove('show'));
    document.querySelectorAll('.term-input').forEach(i => i.classList.remove('active'));
}

// Закрываем при клике в любое пустое место
document.addEventListener('click', closeAllTerm);

// Инициализация всех полей
function initAllDates() {
    initTermInput('tDay', 'day');
    initTermInput('tMonth', 'month');
    initTermInput('tYear', 'year');

    initTermInput('tDay2', 'day');
    initTermInput('tMonth2', 'month');
    initTermInput('tYear2', 'year');
}

function getDate(suffix) {
    // suffix = '' или '2'
    // Берем данные из dataset.val, так как в innerText теперь может быть мусор
    const d = document.getElementById('tDay' + suffix).dataset.val;
    const m = document.getElementById('tMonth' + suffix).dataset.val;
    const y = document.getElementById('tYear' + suffix).dataset.val;
    return `${y}-${m}-${d}`;
}

// === 4. ИНИЦИАЛИЗАЦИЯ ===
document.addEventListener('DOMContentLoaded', () => {
    console.log("App starting [Terminal Mode]...");
    
    // Вместо старого fillDates запускаем новую инициализацию
    initAllDates();
    
    const physics = new SheetPhysics();
    initScrollFx();

    const btnSingle = document.getElementById('btnSingle');
    const btnCompat = document.getElementById('btnCompat');
    const partnerInputs = document.getElementById('partnerInputs');

    // Логика переключения режимов (осталась прежней, только ID учтены)
    if(btnSingle && btnCompat) {
        btnSingle.onclick = (e) => {
            currentMode='single'; 
            if(partnerInputs) partnerInputs.style.display='none'; // Скрываем блок партнера
            btnSingle.classList.add('active'); btnCompat.classList.remove('active');
        };
        btnCompat.onclick = (e) => {
            currentMode='compat'; 
            if(partnerInputs) partnerInputs.style.display='flex'; // Показываем flex для партнера
            btnCompat.classList.add('active'); btnSingle.classList.remove('active');
        };
    }
    const btnCalc = document.getElementById('btnCalculate');
    if(btnCalc) {
        btnCalc.onclick = async () => {
            const originalText = btnCalc.innerText;
            btnCalc.innerText = '...';
            try {
                const dStr = getDate('');
                const d2Str = currentMode==='compat' ? getDate('2') : null;
                
                const res = await renderMatrix(dStr, d2Str);
                globalNodes = res.nodes; 
                globalMeta = res.meta;

                // ВЫЗОВ UI: передаем режим!
                updateUI(currentMode, globalNodes, globalMeta);
                
                // Приоткрываем шторку до 50% (но дальше она свободна)
                physics.openHalf();
                
            } catch(e){ 
                console.error(e); 
                alert("Ошибка расчета: " + e.message);
            }
            btnCalc.innerText = originalText;
        };
    }
    
    // Автозапуск (для теста)
    setTimeout(() => { if(btnCalc) btnCalc.click(); }, 500);
});